// DataWindowPlugIn.cs
//
// This file implements an IDataWindowPlugIn-derived class.
// The plug-in implements the basic parts of a data window
// in the TLA. It has a data display window and a System
// Window Control.
// 
// The window in this sample does not show any real data.
// The purpose is to demonstrate the basics of how to 
// implement the standard parts of a data window. Any custom
// timing diagrams or data interpretations are left as
// an exercise for the plug-in developer.
/////////////////////////////////////////////////////////////

using System;
using System.Windows.Forms;
using Tektronix.LogicAnalyzer.TpiNet;
using Tektronix.LogicAnalyzer.PlugIn;
using Tektronix.LogicAnalyzer.Common;

namespace CSSampleDataWindow
{
	// The PlugInIdentity Attribute used below defines the text
	// used in the New Data Window dialog to identify this kind
	// of data window before an instance of it is created.
	[PlugInIdentity("C# Sample Data Window")]
	public class CSDataWindowPlugIn : IDataWindowPlugIn
	{
    	// Data Members:
		DataDisplay display = null;
		bool isValid = true;
		bool isGarbage = false;
		string userName = string.Empty;
		SystemWindowUserControl control = null;


		// The required default constructor.
		public CSDataWindowPlugIn()
		{
		}
		
	// IPlugIn members

		// Initialize() is called by the TLA immediately after the 
		// plug-in is instantiated with its constructor.
		public bool Initialize(ITlaPlugInSupport support, bool userInit)
		{
			// Create the DataDisplay Form window and move it 
			// into the main window of the application .
			display = new DataDisplay();
			support.TopLevelFormToChild(display);

			// Create the control that will appear in the System Window.
			control = new SystemWindowUserControl();

			// Subscribe to get notified when the data window is closed.
			display.Closed += new EventHandler(OnDisplayClosed);

			// Subscribe to get notified when the "Show Window" button is clicked.
			control.activateButton.Click += new EventHandler(OnShowDataDisplay);
			
			// Subscribe to get notified when the text box loses focus
			control.userNameBox.LostFocus += new EventHandler(OnLostFocus);

			// Subscribe to get notified when keys are pressed in the text box control.
			control.userNameBox.KeyDown += new KeyEventHandler(OnKeyDown);
			
			// Setup to display an intial user name. If the name is not 
			// unique among data windows in the system, the TLA will change it.
			userName = "Sample 1";
			UserName = userName;
			control.userNameBox.Text = userName;
			display.Text = userName;

			return true;
		}

		
		// The primary Form is the data window.
		public Form PrimaryForm
		{
			get 
			{
				return display;
			}
		}

	// IValidity members

		// This event is raised whenever either of the two 
		// properties below change.
		public event EventHandler ValidityChanged;

		
		// Property that indicates whether the plug-in is 
		// valid to be part of the system.
		public bool IsValid
		{
			get 
			{
				return isValid;
			}
		}
		
		
		// Property that indicates whether the plug-in is permanently
		// invalid. This value should always be set to true when a
		// plug-in is disposed.
		public bool IsGarbage 
		{
			get 
			{
				return isGarbage;
			}
		}
		
	
	// IDisposable Member

		// This method invalidates the plug-in and disposes 
		// of its window and user control.
		public void Dispose()
		{
			if (!isGarbage) 
			{
				if (null != display) 
				{
					display.Closed -= new EventHandler(OnDisplayClosed);
					display.Dispose();
					display = null;
				}

				if (null != control) 
				{
					control.activateButton.Click -= 
						new EventHandler(OnShowDataDisplay);
					control.userNameBox.KeyDown -= 
						new KeyEventHandler(OnKeyDown);
					control.userNameBox.LostFocus -= 
						new EventHandler(OnLostFocus);
					control.Dispose();
					control = null;
				}
    		    
				isValid = false;
				isGarbage = true;
    		    
				if (null != ValidityChanged) 
				{
					ValidityChanged(this, EventArgs.Empty);
				}
			}
		}

	
	// IDataWindowPlugIn Members

		// Event that is raised when the UserName property changes.
		public event EventHandler UserNameChanged;
		
		
		// The UserName property sets or gets the name that
		// identifies the data window to users.
		public string UserName
		{
		    get {
		        return userName;
		    }
		    set {
				// First check that the name really changed
				if (0 != string.Compare(userName, value)) 
				{
					userName = value;

					// Update the control and the data window title.
					if (null != control) 
					{
						control.userNameBox.Text = userName;
					}
					if (null != display) 
					{
						display.Text = userName;
					}

					// Send notification that the name changed.
					if (null != UserNameChanged) 
					{
						UserNameChanged(this, EventArgs.Empty);
					}
				}
			}
		}


		// Returns the data windows currently used by the data window.
		// Since this window does not actively use system data sources,
		// it returns an empty array.
		public IDataSource[] GetDataSources() 
		{
			IDataSource[] array;
			array = new IDataSource[0];
			return array;
		}
		
		
		// The value of the SystemWindowControl property is the
		// Control that represents the data window in the
		// TLA System window.
		public UserControl SystemWindowControl
		{
			get 
			{
				return control;
			}
		}

	// Event Handlers

		// Activates the data window when the user presses the "Show Window"
		// button on the SystemWindowUserControl.
		private void OnShowDataDisplay(object sender, EventArgs args) 
		{
			display.Activate();
		}


		// Changes the UserName if the Enter key is pressed in the 
		// SystemWindowUserControl text box.
		private void OnKeyDown(object sender, KeyEventArgs args) 
		{
			if (Keys.Enter == args.KeyCode) 
			{
				UserName = control.userNameBox.Text;
			}
		}


		// Changes the UserName when the SystemWindowUserControl text box
		// loses the input focus.
		private void OnLostFocus(object sender, EventArgs args) 
		{
			UserName = control.userNameBox.Text;
		}


		// The plug-in subscribes to DataDisplay.Closed. This event handler 
		// is called when the event occurs. The plug-in responds by disposing 
		// itself, which deletes the plug-in instance from the system.
		private void OnDisplayClosed(object sender, EventArgs args)
		{
			Dispose();
		}
	}
}
